/*-------------------------------------------------------------------------
 *
 * walsender.c
 *
 *-------------------------------------------------------------------------
 */
#include "postgres.h"

#include <unistd.h>

#include "libpq/pqsignal.h"
#include "miscadmin.h"
#include "postmaster/walsender.h"
#include "storage/bufmgr.h"
#include "storage/fd.h"
#include "storage/ipc.h"
#include "storage/pmsignal.h"
#include "storage/proc.h"
#include "storage/smgr.h"
#include "utils/memutils.h"
#include "utils/ps_status.h"
#include "utils/resowner.h"

/*
 * Shared memory area for communication between walsender and the other.
 */
typedef struct
{
	pid_t	pid;	/* PID of a backend, for requesting */
	
	/* 
	 * Indicates a byte position where walsender is requested to replicate.
	 */
	XLogRecPtr	requestLSN;
} WalSenderRequest;

typedef struct
{
	pid_t	walsender_pid;	/* PID of walsender (0 if not started) */

	bool	replication_active;	/* replication is active? */

	/* Protected by WALSenderCommLock */
	int			num_requests;	/* current # of requests */
	int			max_requests;	/* allocated array size */
	WalSenderRequest requests[1];	/* VARIABLE LENGTH ARRAY */
} WalSenderShmemStruct;

static WalSenderShmemStruct *WalSenderShmem;

/*
 * Global state
 */
bool	am_walsender	= false;

/*
 * Flags set by interrupt handlers for later service in the main loop.
 */
static volatile sig_atomic_t got_SIGHUP = false;
static volatile sig_atomic_t shutdown_requested = false;
static volatile sig_atomic_t replication_requested = false;

/* Signal handlers */
static void RepSigHupHandler(SIGNAL_ARGS);
static void RepTimeoutHandler(SIGNAL_ARGS);
static void RepShutdownHandler(SIGNAL_ARGS);
static void RepQuickDieHandler(SIGNAL_ARGS);
static void RepRequestHandler(SIGNAL_ARGS);

/* Prototypes for private functions */
static int	WalSenderLoop(void);
static void WalSenderShmemCleanup(int, Datum);

/*
 * Main entry point for walsender process
 */
int
WalSenderMain(void)
{
	sigjmp_buf	local_sigjmp_buf;
	MemoryContext walsender_context;

	/*
	 * Check whether this backend is qualified as walsender.
	 */
	if (!am_walsender)
		ereport(FATAL,
				(errcode(ERRCODE_PROTOCOL_VIOLATION),
				 errmsg("unqualified backend received a replication message")));
	
	/*
	 * Set up signal handlers and masks again.
	 */
	pqsignal(SIGHUP, RepSigHupHandler); /* set flag to read config file */
	pqsignal(SIGINT, RepTimeoutHandler);		/* detect timeout */
	pqsignal(SIGTERM, RepShutdownHandler);		/* request shutdown */
	pqsignal(SIGQUIT, RepQuickDieHandler);	/* hard crash time */
	pqsignal(SIGALRM, SIG_IGN);
	pqsignal(SIGPIPE, SIG_IGN);
	pqsignal(SIGUSR1, RepRequestHandler); /* request replication */
	pqsignal(SIGUSR2, SIG_IGN); /* not used */

	/*
	 * Reset some signals
	 */
	pqsignal(SIGCHLD, SIG_DFL);
	pqsignal(SIGTTIN, SIG_DFL);
	pqsignal(SIGTTOU, SIG_DFL);
	pqsignal(SIGCONT, SIG_DFL);
	pqsignal(SIGWINCH, SIG_DFL);

	/* We allow SIGQUIT (quickdie) at all times */
#ifdef HAVE_SIGPROCMASK
	sigdelset(&BlockSig, SIGQUIT);
#else
	BlockSig &= ~(sigmask(SIGQUIT));
#endif

	/*
	 * Create a resource owner to keep track of our resources (not clear that
	 * we need this, but may as well have one).
	 */
	CurrentResourceOwner = ResourceOwnerCreate(NULL, "Wal Sender");

	/*
	 * Create a memory context that we will do all our work in.  We do this so
	 * that we can reset the context during error recovery and thereby avoid
	 * possible memory leaks.  Formerly this code just ran in
	 * TopMemoryContext, but resetting that would be a really bad idea.
	 */
	walsender_context = AllocSetContextCreate(TopMemoryContext,
											  "Wal Sender",
											  ALLOCSET_DEFAULT_MINSIZE,
											  ALLOCSET_DEFAULT_INITSIZE,
											  ALLOCSET_DEFAULT_MAXSIZE);
	MemoryContextSwitchTo(walsender_context);

	/*
	 * Save stack context for handling an exception again.
	 *
	 * This code is heavily based on bgwriter.c, q.v.
	 */
	if (sigsetjmp(local_sigjmp_buf, 1) != 0)
	{
		/* Since not using PG_TRY, must reset error stack by hand */
		error_context_stack = NULL;

		/* Prevent interrupts while cleaning up */
		HOLD_INTERRUPTS();

		/* Report the error to the server log */
		EmitErrorReport();

		/*
		 * These operations are really just a minimal subset of
		 * AbortTransaction().	We don't have very many resources to worry
		 * about in walsender, but we do have LWLocks, and perhaps buffers?
		 */
		LWLockReleaseAll();
		AbortBufferIO();
		UnlockBuffers();
		/* buffer pins are released here: */
		ResourceOwnerRelease(CurrentResourceOwner,
							 RESOURCE_RELEASE_BEFORE_LOCKS,
							 false, true);
		/* we needn't bother with the other ResourceOwnerRelease phases */
		AtEOXact_Buffers(false);
		AtEOXact_Files();
		AtEOXact_HashTables(false);

		/*
		 * Now return to normal top-level context and clear ErrorContext for
		 * next time.
		 */
		MemoryContextSwitchTo(walsender_context);
		FlushErrorState();

		/* Flush any leaked data in the top-level context */
		MemoryContextResetAndDeleteChildren(walsender_context);

		/* Now we can allow interrupts again */
		RESUME_INTERRUPTS();

		/*
		 * Sleep at least 1 second after any error.  A write error is likely
		 * to be repeated, and we don't want to be filling the error logs as
		 * fast as we can.
		 */
		pg_usleep(1000000L);

		/*
		 * Close all open files after any error.  This is helpful on Windows,
		 * where holding deleted files open causes various strange errors.
		 * It's not clear we need it elsewhere, but shouldn't hurt.
		 */
		smgrcloseall();
	}

	/* We can now handle ereport(ERROR) */
	PG_exception_stack = &local_sigjmp_buf;

	/*
	 * Unblock signals (they were blocked when the postmaster forked us)
	 */
	PG_SETMASK(&UnBlockSig);
	
	return WalSenderLoop();
}

/*
 * Main loop of walsender
 */
static int
WalSenderLoop(void)
{
	/*
	 * Loop forever
	 */
	for (;;)
	{
		if (shutdown_requested)
		{
			/* Normal exit from the walsender is here */
			proc_exit(0);		/* done */
		}

		sleep(100);//dummy for this patch
	}
	
	/* can't get here because the above loop never exits */
	return 1;
}

/*
 * Declare that the caller works as walsender after this.
 *
 * This is called only when a database for replication (it's "walsender")
 * is specified in a startup packet.
 */
void
DeclareWalSender(Port *port, char *remote_ps_data)
{
	/*
	 * Check to see if the other walsender still exists.
	 *
	 * If PID of walsender is already set in the shmem, we conclude that
	 * the other walsender still exists. We don't misjudge its existence
	 * because the shmem is initialized whenever walsender dies.
	 *
	 * Now, the number of walsenders is restricted to one.
	 */
	if (WalSenderShmem->walsender_pid != 0)
		ereport(FATAL,
				(errcode(ERRCODE_TOO_MANY_CONNECTIONS),
				 errmsg("sorry, too many walsenders already"),
				 errhint("Is another walsender (PID %d) running?",
						 WalSenderShmem->walsender_pid)));
	
	/*
	 * Set up walsender-exit callback for WalSenderShmem cleanup.
	 *
	 * We have to do this before setting up walsender_pid. Otherwise,
	 * walsender_pid remains set up if walsender dies after setting
	 * walsender_pid before calling on_shmem_exit. This causes a 
	 * mis-judgement of walsender's existence.
	 */
	on_shmem_exit(WalSenderShmemCleanup, 0);

	WalSenderShmem->walsender_pid = getpid();
	
	/*
	 * Set the walsender's command line display in the form:
	 *
	 *     postgres: wal sender process <user> <host> <activity>
	 */
	init_ps_display("wal sender process", port->user_name, remote_ps_data,
					update_process_title ? "authentication" : "");
	
	am_walsender = true;
	
	/*
	 * Declare postmaster that the caller works as background process and
	 * does not need to be treated as a normal backend.
	 */
	SendPostmasterSignal(PMSIGNAL_DECLARE_WALSENDER);
}

/* --------------------------------
 *		signal handler routines
 * --------------------------------
 */

/* SIGHUP: set flag to re-read config file at next convenient time */
static void
RepSigHupHandler(SIGNAL_ARGS)
{
	got_SIGHUP = true;
}

/* SIGINT: terminate walsender due to replication timeout */
static void
RepTimeoutHandler(SIGNAL_ARGS)
{
#if 0
	/*
	 * This would be reviewed in the next patch.
	 */
	ereport(FATAL,
			(errmsg("terminating walsender due to replication timeout"),
			 errdetail("xlog has been replicated to %X/%X, possibly %X/%X",
					   LogsndResult.Recv.xlogid, LogsndResult.Recv.xrecoff,
					   LogsndResult.Send.xlogid, LogsndResult.Send.xrecoff)));
#endif
}

/* SIGTERM: set flag to exit normally */
static void
RepShutdownHandler(SIGNAL_ARGS)
{
	shutdown_requested = true;
}

/* 
 * SIGQUIT: exit quickly
 * 
 * Some backend has bought the farm,
 * so we need to stop what we're doing and exit.
 */
static void
RepQuickDieHandler(SIGNAL_ARGS)
{
	PG_SETMASK(&BlockSig);
	
	/*
	 * DO NOT proc_exit() -- we're here because shared memory may be
	 * corrupted, so we don't want to try to clean up our transaction. Just
	 * nail the windows shut and get out of town.
	 *
	 * Note we do exit(2) not exit(0).	This is to force the postmaster into a
	 * system reset cycle if some idiot DBA sends a manual SIGQUIT to a random
	 * backend.  This is necessary precisely because we don't clean up our
	 * shared memory state.
	 */
	exit(2);
}

/* SIGUSR1: set flag to do replication */
static void
RepRequestHandler(SIGNAL_ARGS)
{
	replication_requested = true;
}

/* --------------------------------
 *		communication with the other process
 * --------------------------------
 */

/*
 * Compute space needed for walsender-related shared memory
 */
Size
WalSenderShmemSize(void)
{
	Size size;
	
	size = offsetof(WalSenderShmemStruct, requests);
	/* auxiiary processes */
	size = add_size(size, mul_size(NUM_AUXILIARY_PROCS, sizeof(WalSenderRequest)));
	/* backends, including autovacuum */
	size = add_size(size, mul_size(MaxBackends, sizeof(WalSenderRequest)));

	return size;
}

/*
 * Allocate and initialize walsender-related shared memory
 */
void
WalSenderShmemInit(void)
{
	bool found;

	WalSenderShmem = (WalSenderShmemStruct *)
		ShmemInitStruct("Wal Sender Data",
						WalSenderShmemSize(),
						&found);
	if (WalSenderShmem == NULL)
		ereport(FATAL,
				(errcode(ERRCODE_OUT_OF_MEMORY),
				 errmsg("not enough shared memory for wal sender")));
	if (found)
		return;					/* already initialized */
	
	MemSet(WalSenderShmem, 0, sizeof(WalSenderShmemStruct));
	WalSenderShmem->max_requests = NUM_AUXILIARY_PROCS + MaxBackends;
}

/*
 * Clean the shmem up.
 *
 * This is called during walsender shutdown.
 */
static void
WalSenderShmemCleanup(int code, Datum arg)
{
	WalSenderShmemStruct *wss = WalSenderShmem;

	LWLockAcquire(WalSenderCommLock, LW_EXCLUSIVE);
	
	wss->walsender_pid = 0;
	wss->replication_active = false;

#if 0
	/*
	 * Disable till signal handling patch is accepted.
	 */
	while (wss->num_requests > 0)
		SendProcSignal(wss->requests[--wss->num_requests].pid, 
					   PROCSIGNAL_COMPLETE_REPLICATION);
#endif

	LWLockRelease(WalSenderCommLock);
}

/*
 * Add the specified request to the shmem.
 *
 * Returns false if replication is already inactive, true otherwise.
 */
bool
WalSenderRequestAdd(pid_t pid, XLogRecPtr requestLSN)
{
	WalSenderShmemStruct *wss = WalSenderShmem;
	WalSenderRequest *request;

	LWLockAcquire(WalSenderCommLock, LW_EXCLUSIVE);
	
	if (!wss->replication_active)
	{
		LWLockRelease(WalSenderCommLock);
		return false;
	}
	
	if (wss->num_requests >= wss->max_requests)
	{
		/* 
		 * No room for new request. (This really shouldn't happen, since 
		 * there is a fixed supply of requests too, and so we should have 
		 * failed earlier.)
		 */
		LWLockRelease(WalSenderCommLock);
		ereport(FATAL,
				(errcode(ERRCODE_TOO_MANY_CONNECTIONS),
				 errmsg("sorry, too many requests from backends already")));
	}
	
	request = &wss->requests[wss->num_requests++];
	request->pid = pid;
	request->requestLSN = requestLSN;
	
	LWLockRelease(WalSenderCommLock);
	return true;
}

/*
 * Remove the request for a given pid from the shmem.
 */
void
WalSenderRequestRemove(pid_t pid)
{
	WalSenderShmemStruct *wss = WalSenderShmem;
	int i;
	
	LWLockAcquire(WalSenderCommLock, LW_EXCLUSIVE);
	
	for (i = 0; i < wss->num_requests; i++)
	{
		if (wss->requests[i].pid == pid)
		{
			wss->requests[i] = wss->requests[--wss->num_requests];
			break;
		}
	}
	
	LWLockRelease(WalSenderCommLock);
}

/* --------------------------------
 *		common routines
 * --------------------------------
 */

/*
 * Get PID of walsender
 */
pid_t
GetWalSenderPid(void)
{
	return WalSenderShmem->walsender_pid;
}
