#include "postgres.h"
#include "funcapi.h"
#include "miscadmin.h"

#include "catalog/pg_type.h"
#include "executor/spi.h"
#include "utils/array.h"
#include "utils/builtins.h"
#include "utils/lsyscache.h"
#include "utils/typcache.h"

Datum	pst_expand_record(PG_FUNCTION_ARGS);
Datum	pst_record_get_field(PG_FUNCTION_ARGS);
Datum	pst_record_set_field(PG_FUNCTION_ARGS);

PG_FUNCTION_INFO_V1(pst_expand_record);
PG_FUNCTION_INFO_V1(pst_record_get_field);
PG_FUNCTION_INFO_V1(pst_record_set_field);


/***
 * FUNCTION expand_record(record, 
 * 				    OUT name text, OUT value, OUT typ text)
 * 	RETURNS SETOF record
 ***
 */
Datum 
pst_expand_record(PG_FUNCTION_ARGS)
{
	ReturnSetInfo *rsinfo = (ReturnSetInfo *) fcinfo->resultinfo;
	TupleDesc rstupdesc; 
	Datum	values[3];
	bool	nulls[3] = {false, false, false};

	HeapTupleHeader		rec = PG_GETARG_HEAPTUPLEHEADER(0);
	TupleDesc		rectupdesc;
	Oid			rectuptyp;
	int32			rectuptypmod;
	HeapTupleData		rectuple;
	int	ncolumns;
	Datum 		*recvalues;
	bool  		*recnulls;

	Tuplestorestate *tupstore;		/* output tuplestore */
	MemoryContext per_query_ctx;
	MemoryContext oldcontext;
	HeapTuple	tuple;
	int			i;

	/* check to see if caller supports us returning a tuplestore */
	if (rsinfo == NULL || !IsA(rsinfo, ReturnSetInfo))
		ereport(ERROR,
				(errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
				 errmsg("set-valued function called in context that cannot accept a set")));
	if (!(rsinfo->allowedModes & SFRM_Materialize))
		ereport(ERROR,
				(errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
				 errmsg("materialize mode required, but it is not " \
						"allowed in this context")));

	/* Extract type info from the tuple itself */
	rectuptyp = HeapTupleHeaderGetTypeId(rec);
	rectuptypmod = HeapTupleHeaderGetTypMod(rec);
	rectupdesc = lookup_rowtype_tupdesc(rectuptyp, rectuptypmod);
	ncolumns = rectupdesc->natts;

	/* Build a temporary HeapTuple control structure */
	rectuple.t_len = HeapTupleHeaderGetDatumLength(rec);
	ItemPointerSetInvalid(&(rectuple.t_self));
	rectuple.t_tableOid = InvalidOid;
	rectuple.t_data = rec;

	recvalues = (Datum *) palloc(ncolumns * sizeof(Datum));
	recnulls = (bool *) palloc(ncolumns * sizeof(bool));
	
	/* Break down the tuple into fields */
	heap_deform_tuple(&rectuple, rectupdesc, recvalues, recnulls);

	/* need to build tuplestore in query context */
	per_query_ctx = rsinfo->econtext->ecxt_per_query_memory;
	oldcontext = MemoryContextSwitchTo(per_query_ctx);
	tupstore = tuplestore_begin_heap(true, false, work_mem);
	rstupdesc = CreateTupleDescCopy(rsinfo->expectedDesc);

	for (i = 0; i < ncolumns; i++)
	{
		Oid	columntyp = rectupdesc->attrs[i]->atttypid;
		int32	columntypmod = rectupdesc->attrs[i]->atttypmod;


		/* Ignore dropped columns */
		if (rectupdesc->attrs[i]->attisdropped)
			continue;

		/* generate junk in short-term context */
		MemoryContextSwitchTo(oldcontext);
                                                            
		values[0] = CStringGetTextDatum(NameStr(rectupdesc->attrs[i]->attname));
		values[2] = CStringGetTextDatum(format_type_with_typemod(columntyp, columntypmod));

		if (!recnulls[i])
		{
			char *outstr;
			bool		typIsVarlena;
			Oid		typoutput;
			FmgrInfo		proc;
			
			getTypeOutputInfo(columntyp, &typoutput, &typIsVarlena);
			fmgr_info_cxt(typoutput, &proc, oldcontext);
			outstr = OutputFunctionCall(&proc, recvalues[i]);

			values[1] = CStringGetTextDatum(outstr);
			nulls[1] = false;
		}
		else
			nulls[1] = true;

		tuple = heap_form_tuple(rstupdesc, values, nulls);

		MemoryContextSwitchTo(per_query_ctx);
		tuplestore_puttuple(tupstore, tuple);
	}

	ReleaseTupleDesc(rectupdesc);

	/* clean up and return the tuplestore */
	tuplestore_donestoring(tupstore);
        
	MemoryContextSwitchTo(oldcontext);
                
	rsinfo->returnMode = SFRM_Materialize;
	rsinfo->setResult = tupstore;
	rsinfo->setDesc = rstupdesc;
        
        pfree(recvalues);
        pfree(recnulls);
        
	return (Datum) 0;
}

/***
 * FUNCTION record_get_field(record) RETURNS text 
 *
 ***
 */
Datum
pst_record_get_field(PG_FUNCTION_ARGS)
{
	HeapTupleHeader		rec = PG_GETARG_HEAPTUPLEHEADER(0);
	TupleDesc		rectupdesc;
	Oid			rectuptyp;
	int32			rectuptypmod;
	HeapTupleData		rectuple;
	int	fno;
	char *outstr;
	bool		typIsVarlena;
	Oid		typoutput;
	FmgrInfo		proc;
	Oid	typeid;
	Datum		value;
	bool		isnull;

	char	*fieldname = text_to_cstring(PG_GETARG_TEXT_P(1));

	/* Extract type info from the tuple itself */
	rectuptyp = HeapTupleHeaderGetTypeId(rec);
	rectuptypmod = HeapTupleHeaderGetTypMod(rec);
	rectupdesc = lookup_rowtype_tupdesc(rectuptyp, rectuptypmod);

	/* Build a temporary HeapTuple control structure */
	rectuple.t_len = HeapTupleHeaderGetDatumLength(rec);
	ItemPointerSetInvalid(&(rectuple.t_self));
	rectuple.t_tableOid = InvalidOid;
	rectuple.t_data = rec;

	fno = SPI_fnumber(rectupdesc, fieldname);
	if (fno == SPI_ERROR_NOATTRIBUTE)
				ereport(ERROR,
						(errcode(ERRCODE_UNDEFINED_COLUMN),
						 errmsg("record has no field \"%s\"",
								fieldname)));

	value = SPI_getbinval(&rectuple, rectupdesc, fno, &isnull);
	if (isnull)
	{
		ReleaseTupleDesc(rectupdesc);
		PG_RETURN_NULL();
	}

	typeid = SPI_gettypeid(rectupdesc, fno);
	getTypeOutputInfo(typeid, &typoutput, &typIsVarlena);
	fmgr_info(typoutput, &proc);
	outstr = OutputFunctionCall(&proc, value);

	ReleaseTupleDesc(rectupdesc);

	PG_RETURN_TEXT_P(cstring_to_text(outstr));
}

/***
 * FUNCTION record_set_field(anyelement, VARIADIC params text[])
 * 									RETURNS anyelement AS 
 ***
 */
Datum
pst_record_set_field(PG_FUNCTION_ARGS)
{
	HeapTupleHeader		rec;
	HeapTupleHeader		result;
	TupleDesc	rectupdesc;
	Oid         rectuptyp = get_fn_expr_argtype(fcinfo->flinfo, 0);
	int32			rectuptypmod;
	HeapTupleData		rectuple;
	int	ncolumns;
	Datum 		*recvalues;
	bool  		*recnulls;
	HeapTuple	tuple;

	int		i;
	char			*fieldname = NULL;

	if (rectuptyp == RECORDOID)
	{
		rec = PG_GETARG_HEAPTUPLEHEADER(0);

		/* Extract type info from the tuple itself */
		rectuptyp = HeapTupleHeaderGetTypeId(rec);
		rectuptypmod = HeapTupleHeaderGetTypMod(rec);
		rectupdesc = lookup_rowtype_tupdesc(rectuptyp, rectuptypmod);
	}
	else 
	{
		rectuptypmod = -1;
		rectupdesc = lookup_rowtype_tupdesc(rectuptyp, rectuptypmod);
		rec = PG_GETARG_HEAPTUPLEHEADER(0);
	}

	ncolumns = rectupdesc->natts;

	/* Build a temporary HeapTuple control structure */
	rectuple.t_len = HeapTupleHeaderGetDatumLength(rec);
	ItemPointerSetInvalid(&(rectuple.t_self));
	rectuple.t_tableOid = InvalidOid;
	rectuple.t_data = rec;

	recvalues = (Datum *) palloc(ncolumns * sizeof(Datum));
	recnulls = (bool *) palloc(ncolumns * sizeof(bool));
	
	/* Break down the tuple into fields */
	heap_deform_tuple(&rectuple, rectupdesc, recvalues, recnulls);

	/* process a variadic arguments - key and newval */

	if (PG_NARGS() % 2 != 1)
		elog(ERROR, "missing a value parameter");

	for (i = 1; i < PG_NARGS(); i++)
	{
		bool	isnull;
		Datum	value;
		char		*cstr;

		if (PG_ARGISNULL(i))
		{
			isnull = true;
			cstr = NULL;
		}
		else
		{
			Oid valtype;
			bool		typIsVarlena;
			Oid		typoutput;
			FmgrInfo		proc;

			isnull = false;
			value = PG_GETARG_DATUM(i);

			valtype = get_fn_expr_argtype(fcinfo->flinfo, i);
			getTypeOutputInfo(valtype, &typoutput, &typIsVarlena);
			fmgr_info(typoutput, &proc);
			cstr = OutputFunctionCall(&proc, value);
		}

		/* if parameter is key, store name, else convert value to target type */
		if (i % 2 == 1)
		{
			if (isnull)
				elog(ERROR, "field name is null");

			fieldname = cstr; 
		}
		else
		{
			int fno;
			
			fno = SPI_fnumber(rectupdesc, fieldname);
			if (fno == SPI_ERROR_NOATTRIBUTE)
					ereport(ERROR,
							(errcode(ERRCODE_UNDEFINED_COLUMN),
							 errmsg("record has no field \"%s\"",
											fieldname)));

			if (isnull)
			{
				recnulls[fno - 1] = true;
				pfree(fieldname);
			}
			else
			{
				Oid	typeid;
				Oid		typiofunc;
				Oid		typioparam;
				FmgrInfo	proc;

		    		typeid = SPI_gettypeid(rectupdesc, fno);
				getTypeInputInfo(typeid, &typiofunc, &typioparam);
				fmgr_info(typiofunc, &proc);
				recvalues[fno - 1] = InputFunctionCall(&proc, cstr, typioparam, 
											rectupdesc->attrs[fno - 1]->atttypmod);
				recnulls[fno - 1] = false;

				pfree(fieldname);
				pfree(cstr);
			}
		} 
	}

	tuple = heap_form_tuple(rectupdesc, recvalues, recnulls);

	/*
	 * We cannot return tuple->t_data because heap_form_tuple allocates it as
	 * part of a larger chunk, and our caller may expect to be able to pfree
	 * our result.	So must copy the info into a new palloc chunk.
	 */
	result = (HeapTupleHeader) palloc(tuple->t_len);
	memcpy(result, tuple->t_data, tuple->t_len);

	heap_freetuple(tuple);
	pfree(recvalues);
	pfree(recnulls);
	ReleaseTupleDesc(rectupdesc);

	PG_RETURN_HEAPTUPLEHEADER(result);
}

/*
 * last function should be overloaded, because there are bug for variadic "any"
 */
