#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include "varchar.h"

#define BUF_SIZE	1073741824	// A gig worth of memory.
#define COUNT		100000		// Number of iterations to perform.
#define NUM_TRIALS	7			// Number of trials for speed test.
								// Return median of the trials.

unsigned char *buffer;

static unsigned long int seed = 0;
static unsigned int permute[] = {
	0x01234, 0x10234, 0x02134, 0x12034, 0x20134, 0x21034,
	0x01324, 0x10324, 0x02314, 0x12304, 0x20314, 0x21304,
	0x03124, 0x13024, 0x03214, 0x13204, 0x23014, 0x23104,
	0x30124, 0x31024, 0x30214, 0x31204, 0x32014, 0x32104,

	0x01243, 0x10243, 0x02143, 0x12043, 0x20143, 0x21043,
	0x01342, 0x10342, 0x02341, 0x12340, 0x20341, 0x21340,
	0x03142, 0x13042, 0x03241, 0x13240, 0x23041, 0x23140,
	0x30142, 0x31042, 0x30241, 0x31240, 0x32041, 0x32140,

	0x01423, 0x10423, 0x02413, 0x12403, 0x20413, 0x21403,
	0x01432, 0x10432, 0x02431, 0x12430, 0x20431, 0x21430,
	0x03412, 0x13402, 0x03421, 0x13420, 0x23401, 0x23410,
	0x30412, 0x31402, 0x30421, 0x31420, 0x32401, 0x32410,

	0x04123, 0x14023, 0x04213, 0x14203, 0x24013, 0x24103,
	0x04132, 0x14032, 0x04231, 0x14230, 0x24031, 0x24130,
	0x04312, 0x14302, 0x04321, 0x14320, 0x24301, 0x24310,
	0x34012, 0x34102, 0x34021, 0x34120, 0x34201, 0x34210,

	0x40123, 0x41023, 0x40213, 0x41203, 0x42013, 0x42103,
	0x40132, 0x41032, 0x40231, 0x41230, 0x42031, 0x42130,
	0x40312, 0x41302, 0x40321, 0x41320, 0x42301, 0x42310,
	0x43012, 0x43102, 0x43021, 0x43120, 0x43201, 0x43210
};

/*
 * Return a "random" number between 0 and 2^32-1 inclusive
 */
unsigned int next_random(void)
{
	seed = seed * 1103515245 + 12345;
	return (unsigned int)(seed);
}

/*
 * Populate the memory buffer with test cases for the
 * various "BpTruelen" functions.
 * For this routine, the single byte header format is being
 * used.
 * Inputs
 *    offset = value from 0 to 3 for the beginning of the test case.
 *    maxlen = maximum length of data structure.
 *    			values range from 1 and up.
 *    truelen = number of non-blank characters at beginning of data,
 *    			remaining characters in structure are blank.
 *    			values are <= maxlen;
 */
void init_buffer_1b(int offset, int maxlen, int truelen)
{
	long	x;
	int		y;
	int		size;
	int		limit;
	unsigned char *p;

	size = maxlen + 8;
	size &= 0xfffffffc;
	limit = BUF_SIZE / size;

	for(x=0; x<limit; ++x) {
		p = buffer + x*size + offset;
		*p = (maxlen+1)*2+1;
		++p;
		for(y=0; y<maxlen; ++y) {
			p[y] = (y < truelen) ? 'a' : ' ';
		}
	}
}

/*
 * Populate the memory buffer with test cases for the
 * various "BpTruelen" functions.
 * For this routine, the four byte header format is being
 * used.
 * Inputs
 *    maxlen = maximum length of data structure.
 *    			values range from 1 to 28.
 *    truelen = number of non-blank characters at beginning of data,
 *    			remaining characters in structure are blank.
 *    			values are <= maxlen;
 * A total of 16777216 test structures will be initialized.
 */
void init_buffer_4b(int maxlen, int truelen)
{
	long	x;
	int		y;
	unsigned char *p;
	int		size;
	int		limit;

	size = maxlen + 8;
	size &= 0xfffffffc;
	limit = BUF_SIZE / size;

	for(x=0; x<limit; ++x) {
		p = buffer + x*size;
		*((unsigned long *)p) = (maxlen+4)*4;
		p += 4;
		for(y=0; y<maxlen; ++y) {
			p[y] = (y < truelen) ? 'a' : ' ';
		}
	}
}

long long do_test_2(int offset, int size, int limit, int match, int (*func)(BpChar *))
{
	struct timespec start;
	struct timespec stop;
	long long start_nano;
	long long stop_nano;
	void *p;
	long count;

	if (clock_gettime(CLOCK_REALTIME, &start))
	{
		fprintf(stderr, "Unable to get start time.\n");
		exit(EXIT_FAILURE);
	}
	for(count=0; count < COUNT; ++count)
	{
		p = buffer +  (next_random() % limit)*size + offset;
		if (func(p) != match) {
			fprintf(stderr, "Incorrect length returned.\n");
			exit(EXIT_FAILURE);
		}
	}
	if (clock_gettime(CLOCK_REALTIME, &stop))
	{
		fprintf(stderr, "Unable to get stop time.\n");
		exit(EXIT_FAILURE);
	}
	start_nano = start.tv_sec * 1000000000LL + start.tv_nsec;
	stop_nano  = stop.tv_sec  * 1000000000LL + stop.tv_nsec;
	return (stop_nano - start_nano);
}

int compare(const void *s1, const void *s2)
{
	const long long *p1 = s1;
	const long long *p2 = s2;

	if (*p1 < *p2)
		return -1;
	else if (*p1 > *p2)
		return 1;
	return 0;
}

long long do_test(int offset, int maxlen, int match, int (*func)(BpChar *))
{
	int size;
	int limit;
	long long sample[NUM_TRIALS];
	int x;

	size = maxlen + 8;
	size &= 0xfffffffc;
	limit = BUF_SIZE / size;

	/* Collect some timing sample */
	for(x=0; x<NUM_TRIALS; ++x) {
		sample[x] = do_test_2(offset, size, limit, match, func);
	}

	/* Return the median timing value */
	qsort(sample, NUM_TRIALS, sizeof(sample[0]), compare);
	return (sample[NUM_TRIALS/2] + sample[(NUM_TRIALS-1)/2]) / 2;
}

void init(void)
{
	buffer = malloc(BUF_SIZE);
	if (buffer == NULL)
	{
		fprintf(stderr, "BUF_SIZE too large.\n");
		exit(EXIT_FAILURE);
	}
}

int main(void)
{
	struct timespec res;
	int offset;
	int maxlen;
	int truelen;
	int cycle;
	int format;
	int	pp;
	unsigned int perm;
	long long time1;
	long long time2;
	long long time3;
	long long time4;
	long long overhead;

	if (clock_getres(CLOCK_REALTIME, &res))
	{
		fprintf(stderr, "Unable to determine clock resolution.\n");
		exit(EXIT_FAILURE);
	}
	
	printf("Clock resolution = %lld nanoseconds\n",
		res.tv_sec * 1000000000LL + res.tv_nsec);

	init();

	printf("hdr,offset,maxlen,spaces,overhead,by1,by4,by4sentinel,by1sentinel\n");
	cycle = 0;
	for(maxlen=1; maxlen <= 256; ++maxlen)
	{
		for(truelen=maxlen; truelen >= 0; --truelen)
		{
			for(format=1; format < 5; format += 3)
			{
				for(offset=0; offset<4; ++offset)
				{
					if (format == 1)
					{
						if (maxlen > 126)
							continue;
						init_buffer_1b(offset, maxlen, truelen);
					}
					else
					{
						if (offset != 0)
							continue;
						init_buffer_4b(maxlen, truelen);
					}
					/*
					 * Vary the call order for the test functions
					 * so none of them get a possible long term
					 * advantage over the others from caching.
					 */
					perm=permute[cycle];
					cycle = (cycle+31)%120;
					for(pp=0; pp<5; ++pp)
					{
						switch(perm & 0x7)
						{
							case 0x0:
								overhead = do_test(offset, maxlen, 0, bcNop);
								break;
							case 0x1:
								time1 = do_test(offset, maxlen, truelen, bcTruelen1);
								break;
							case 0x2:
								time2 = do_test(offset, maxlen, truelen, bcTruelen2);
								break;
							case 0x3:
								time3 = do_test(offset, maxlen, truelen, bcTruelen3);
								break;
							case 0x4:
								time4 = do_test(offset, maxlen, truelen, bcTruelen4);
								break;
						}
						perm >>= 4;
					}

					printf("%d,%d,%3d,%3d,%8lld,%10lld,%10lld,%10lld,%10lld\n",
						   format, offset, maxlen, maxlen-truelen, overhead,
						   time1, time2, time3, time4);
					fflush(stdout);
				}
			}
		}
	}
	return 0;
}
