#!/bin/sh

## common
PGDIR=$(pwd)          # postgres installation directory
PGBIN=${PGDIR}/bin
PGBKP=${PGDIR}/pgbkp  # location of a base-backup

PROGNAME=$(basename ${0})    # name of this script

## template of configuration file
PGCONF_TEMPLATE=${PGDIR}/postgresql.conf.template
PG_HBA_TEMPLATE=${PGDIR}/pg_hba.conf.template
RECOVERY_TEMPLATE=${PGDIR}/recovery.conf.template

## about the primary
ACTDATA=${PGDIR}/actdata
ACTARCH=${PGDIR}/actarch
ACTHOST=localhost
ACTPORT=5432
ACTUSER=postgres

## about the standby
SBYDATA=${PGDIR}/sbydata
SBYARCH=${PGDIR}/sbyarch
SBYHOST=localhost
SBYPORT=5433
SBYUSER=postgres

NUM_SBYS=1            # number of standbys that this script starts
NUM_RESERVED_SBYS=5   # number of reserved standbys

## for recovery.conf
CONNINFO="host=${ACTHOST} port=${ACTPORT} user=${ACTUSER}"
TRIGGER=${PGDIR}/trigger

## options
USE_PGARCH=FALSE
USE_PGXLOG=FALSE
USE_NEWBKP=FALSE
SETUP_ACT=TRUE
SLEEPTIME=1

## show usage
usage ()
{
    echo "${PROGNAME} builds the environment for streaming replication"
    echo ""
    echo "Usage:"
    echo "  ${PROGNAME} [OPTIONS]"
    echo ""
    echo "Default:"
    echo "  * sets up one primary and one standby"
    echo "  * empties pg_xlog before starting standby"
    echo "  * doesn't make standby use archived files"
    echo ""
    echo "Options:"
    echo "  -a            copies all archived files from primary to standbys before starting them,"
    echo "                specifies restore_command and makes them use archived files"
    echo "  -b            makes a fresh base backup and uses it for new standbys"
    echo "  -h            shows this help, then exits"
    echo "  -n NUM        number of standbys (default: 1)"
    echo "  -q            quits all servers immediately (i.e., immediate shutdown), then exits"
    echo "  -s            adds new standbys into current environment, then exits"
    echo "  -w SLEEPTIME  seconds to wait between starts of standby (default: 1)"
    echo "  -x            doesn't empty pg_xlog before starting standbys"
}

## quit all servers immediately
quit_all_servers ()
{
    ### quit primary
    ${PGBIN}/pg_ctl -D ${ACTDATA} -mi stop > /dev/null 2>&1

    ### quit standbys
    for _SBYDATA in $(ls -d ${SBYDATA}*); do
	${PGBIN}/pg_ctl -D ${_SBYDATA} -mi stop > /dev/null 2>&1
    done
}

## set up one primary
setup_one_primary ()
{
    ### is primary already in progress?
    if [ -e ${ACTDATA} ]; then
	${PGBIN}/pg_ctl -D ${ACTDATA} status > /dev/null 2>&1
	if [ ${?} -eq 0 ]; then
	    echo "primary is already in progress; cannot start new primary unless shutting down old one"
	    exit 1
	fi
    fi

    ### ensure that there are no standbys in progress
    quit_all_servers

    ### remove old data
    rm -rf ${ACTDATA}* ${ACTARCH}* ${SBYDATA}* ${SBYARCH}* ${TRIGGER}*
    rm -rf ${PGBKP} ${PGCONF_TEMPLATE} ${PG_HBA_TEMPALTE} ${RECOVERY_TEMPLATE}

    ### initialize database cluster and create archive directory
    ${PGBIN}/initdb -D ${ACTDATA} --locale=C --encoding=UTF8
    mkdir ${ACTARCH}

    ### create template of postgresql.conf
    MAX_WAL_SNDS=$(expr ${NUM_SBYS} + ${NUM_RESERVED_SBYS})
    cp ${ACTDATA}/postgresql.conf ${PGCONF_TEMPLATE}
    echo "listen_addresses    = '*'"             >> ${PGCONF_TEMPLATE}
    echo "checkpoint_segments = 32"              >> ${PGCONF_TEMPLATE}
    echo "archive_mode        = on"              >> ${PGCONF_TEMPLATE}
    echo "max_wal_senders     = ${MAX_WAL_SNDS}" >> ${PGCONF_TEMPLATE}
    cp -f ${PGCONF_TEMPLATE} ${ACTDATA}/postgresql.conf

    ### create template of pg_hba.conf; allows all connections
    cp ${ACTDATA}/pg_hba.conf ${PG_HBA_TEMPLATE}
    echo "host all all 0.0.0.0/0 trust"       >> ${PG_HBA_TEMPLATE}
    cp -f ${PG_HBA_TEMPLATE} ${ACTDATA}/pg_hba.conf

    ### create template of recovery.conf
    echo "standby_mode     = 'true'      "  >> ${RECOVERY_TEMPLATE}
    echo "primary_conninfo = '${CONNINFO}'" >> ${RECOVERY_TEMPLATE}

    ### add settins only for primary
    echo "port            = ${ACTPORT}"            >> ${ACTDATA}/postgresql.conf
    echo "archive_command = 'cp %p ${ACTARCH}/%f'" >> ${ACTDATA}/postgresql.conf
    echo "log_line_prefix = 'Act  [%p]: '"         >> ${ACTDATA}/postgresql.conf

    ### start primary
    ${PGBIN}/pg_ctl -D ${ACTDATA} -w start
}

## set up one standby
setup_one_standby ()
{
    ### is primary still in progress?
    if [ ! -e ${ACTDATA} ]; then
	echo "primary is not in progress; cannot start new standby unless starting up primary"
	exit 1
    fi
    ${PGBIN}/pg_ctl -D ${ACTDATA} status > /dev/null 2>&1
    if [ ${?} -ne 0 ]; then
	echo "primary is not in progress; cannot start new standby unless starting up primary"
	exit 1
    fi

    ### get id of this standby
    _SBYID=${1}

    ### calculate this standby's settings
    _SBYDATA=${SBYDATA}${_SBYID}
    _SBYARCH=${SBYARCH}${_SBYID}
    _SBYPORT=$(expr ${SBYPORT} + ${_SBYID})
    _TRIGGER=${TRIGGER}${_SBYID}

    ### create database cluster by copying base-backup
    cp -r ${PGBKP} ${_SBYDATA}
    mkdir ${_SBYARCH}

    ### empty pg_xlog
    if [ "${USE_PGXLOG}" == "FALSE" ]; then
	rm -rf ${_SBYDATA}/pg_xlog/*
    fi

    ### configuration
    cp -f ${PGCONF_TEMPLATE}   ${_SBYDATA}/postgresql.conf
    cp -f ${PG_HBA_TEMPLATE}   ${_SBYDATA}/pg_hba.conf
    cp -r ${RECOVERY_TEMPLATE} ${_SBYDATA}/recovery.conf

    echo "port            = ${_SBYPORT}"            >> ${_SBYDATA}/postgresql.conf
    echo "archive_command = 'cp %p ${_SBYARCH}/%f'" >> ${_SBYDATA}/postgresql.conf
    echo "log_line_prefix = 'Sby${_SBYID}  [%p]: '" >> ${_SBYDATA}/postgresql.conf
    echo "trigger_file    = '${_TRIGGER}'"          >> ${_SBYDATA}/recovery.conf

    ### use archived file?
    if [ "${USE_PGARCH}" == "TRUE" ]; then
	echo "restore_command = 'cp ${_SBYARCH}/%f %p'" >> ${_SBYDATA}/recovery.conf
	cp ${ACTARCH}/* ${_SBYARCH}    ### copy all archived file to standby
    fi

    ### start standby
    ${PGBIN}/pg_ctl -D ${_SBYDATA} start
}

## add new standbys
add_new_standbys ()
{
    SBYID=0

    for i in $(seq 1 ${NUM_SBYS}); do
	### find free id for standby
	while [ 1 ]
	  do
	  _SBYDATA=${SBYDATA}${SBYID}
	  if [ ! -e ${_SBYDATA} ]; then
	      break    ### found!
	  fi

	  SBYID=$(expr ${SBYID} + 1)
	done

	### set up standby
	setup_one_standby ${SBYID}

	### sleep before next setup
	if [ ${i} -lt ${NUM_SBYS} ]; then
	    sleep ${SLEEPTIME}
	fi
    done
}

## make a base-backup
make_base_backup ()
{
    ### online-backup
    ${PGBIN}/psql template1 -c "SELECT pg_start_backup('replication', true)"
    rsync -a --exclude=postmaster.pid ${ACTDATA}/ ${PGBKP}
    ${PGBIN}/psql template1 -c "SELECT pg_stop_backup()"
}

####### main ########

## check if we are in the right place
if [ ! -f ${PGBIN}/pg_config ]; then
    echo "${PGDIR} is not the postgres installation directory"
    exit 1
fi

## parse the command-line arguments
while getopts "abhn:qsw:x" OPT; do
    case ${OPT} in
	a)
	    USE_PGARCH=TRUE
	    ;;
	b)
	    USE_NEWBKP=TRUE
	    ;;
	h)
	    usage
	    exit 0
	    ;;
	n)
	    NUM_SBYS=${OPTARG}
	    ;;
	q)
	    quit_all_servers
	    exit 0
	    ;;
	s)
	    SETUP_ACT=FALSE
	    ;;
	w)
	    SLEEPTIME=${OPTARG}
	    ;;
	x)
	    USE_PGXLOG=TRUE
	    ;;
    esac
done
shift $((${OPTIND} - 1))

## set up primary if requested
if [ "${SETUP_ACT}" == "TRUE" ]; then
    setup_one_primary
fi

## make a fresh base backup for new standbys
if [ "${SETUP_ACT}" == "TRUE" -o "${USE_NEWBKP}" == "TRUE" ]; then
    make_base_backup
fi

## set up standbys
add_new_standbys
