#include "postgres.h"

#include <signal.h>
#include <unistd.h>

#include "libpq-fe.h"
#include "pqsignal.h"

#include "access/xlog_internal.h"

#ifdef WIN32
int			getopt(int argc, char *const argv[], const char *optstring);
#else
#include <sys/time.h>
#include <unistd.h>

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif
#endif   /* ! WIN32 */

extern char *optarg;
extern int	 optind;

/*
 * Options and defaults
 */

/* Communication */
static char	*repHost	= NULL;
static char	*repPort	= NULL;
static char *repOptions	= NULL;
static char *repTty		= NULL;
static char *repDbName	= "postgres";
static char	*repLogin	= NULL;
static char *repPwd		= NULL;

/* Xlog writing */
static char *xlogLoc = ".";	 /* location where xlog is written in */
static char *archLoc = NULL; /* location where filled segment is archived in */

/*
 * Private variables
 */
static const char  *progname;	/* Program name ("walreceiver") */
static PGconn	   *repConn;	/* Connection handle to DB */
static TimeLineID	repTLI = 0;	/* Current xlog timeline */

/*
 * writeFile is -1 or a kernel FD for an open log file segment.
 * When it's open, writeOff is the current seek offset in the file.
 * writeId/writeSeg identify the segment.  These variables are only
 * used to write the xlog, and so will normally refer to the active
 * segment.
 */
static int		writeFile	= -1;
static uint32	writeId		= 0;
static uint32	writeSeg	= 0;
static uint32	writeOff	= 0;

/* Buffer for currenly received xlog */
static char	   *recvBuf = NULL;
static size_t	bufSize = 0;

/*
 * Macros
 */
#define RepXLogFilePath(path, dir, tli, log, seg)						\
	snprintf(path, MAXPGPATH, "%s/%08X%08X%08X", dir, tli, log, seg)

#define RepXLogArchivingActive() (archLoc != NULL)

/* Signal handlers */
static void RepExitHandler(SIGNAL_ARGS);

/*
 * Prototypes for private functions
 */
static int		WalReceiverLoop(void);
static void		RepExit(int);
static int		RepXLogFileInit(uint32, uint32);
static int		RepXLogFileOpen(uint32, uint32);
static void		RepXLogFileClose(void);
static void		RepXLogWrite(XLogRecPtr, int32);
static void		RepXLogArchive(void);
static PGconn  *RepConnect(void);
static void		RepGetBytes(char *, size_t, PGconn *);
static void		RepPutMessage(PGconn *, char, const void *, size_t);
static int		RepCheckRecvBufSpace(size_t);
static void		usage(void);

/*
 * Main entry point for walreceiver
 */
int
main(int argc, char **argv)
{
	int		c;
	char   *env;
	
#ifdef WIN32
	/* stderr is buffered on Win32. */
	setvbuf(stderr, NULL, _IONBF, 0);
#endif
	
	/*
	 * Set up signal handlers
	 */
	pqsignal(SIGINT,  RepExitHandler); /* normal exit */
	pqsignal(SIGTERM, RepExitHandler); /* normal exit */
	pqsignal(SIGQUIT, RepExitHandler); /* normal exit */

	progname = get_progname(argv[0]);
	
	if ((env = getenv("PGHOST")) != NULL && *env != '\0')
		repHost = env;
	if ((env = getenv("PGPORT")) != NULL && *env != '\0')
		repPort = env;
	else if ((env = getenv("PGUSER")) != NULL && *env != '\0')
		repLogin = env;
	
	while ((c = getopt(argc, argv, "h:p:U:?")) != -1)
	{
		switch (c)
		{
			case 'h':
				repHost = optarg;
				break;
			case 'p':
				repPort = optarg;
				break;
			case 'U':
				repLogin = optarg;
				break;
			case '?':
				usage();
				exit(0);
		}
	}
	
	/*
	 * xlogLoc and archLoc might not exist, which would not be an error.
	 * If so, we use current directory as xlog location, and don't archive
	 * xlog.
	 */
	if (optind < argc)
	{
		xlogLoc = argv[optind];
		optind++;
	}
	
	if (optind < argc)
	{
		archLoc = argv[optind];
		optind++;
	}
	
	
	if (recvBuf == NULL)
	{
		recvBuf = (char *) calloc(XLOG_BLCKSZ, sizeof(char));
		if (recvBuf == NULL)
		{
			fprintf(stderr, _("%s: could not allocate memory for"
							  "input buffer\n"), progname);
			exit(1);
		}
		bufSize = XLOG_BLCKSZ;
	}
	
	return WalReceiverLoop();
}

/*
 * Main loop of walreceiver
 */
static int
WalReceiverLoop(void)
{
	repConn = RepConnect();

	/* Send mimic message */
	{
		char mimic_subtype = 'R'; /* means replication */
		RepPutMessage(repConn, 'm', (const void *) &mimic_subtype, 1);
	}
	
	/*
	 * Loop forever
	 */
	for (;;)
	{
		char	qtype;
		int32	len;

		qtype = PQgetByte(repConn);
		if (qtype == EOF)
		{
			fprintf(stderr, _("%s: unexpected EOF within message type: %s\n"),
					progname, PQerrorMessage(repConn));
			RepExit(1);
		}

		RepGetBytes((char *) &len, 4, repConn);
		
		len = ntohl(len) - 4;
		if (len < 0)
		{
			fprintf(stderr, _("%s: invalid message length: %d\n"), 
					progname, len);
			RepExit(1);
		}
		
		switch (qtype)
		{
			case 'l':	/* timeline ID */
			{
				RepGetBytes((char *) &repTLI, 4, repConn);
				repTLI = ntohl(repTLI);

				break;
			}
				
			case 'w':	/* xlog */
			{
				XLogRecPtr recptr;
				XLogRecPtr WriteRqst;
				
				RepGetBytes((char *) &recptr, 8, repConn);
				WriteRqst.xlogid	= ntohl(recptr.xlogid);
				WriteRqst.xrecoff	= ntohl(recptr.xrecoff);
				len -= 8;

				/* Check whether recvBuf has enough room to receive xlog */
				if (RepCheckRecvBufSpace(len) == EOF)
				{
					fprintf(stderr, _("%s: could not allocate memory for"
									  "input buffer\n"), progname);
					RepExit(1);
				}

				/*
				 * Receive xlog, reply a replication-complete message and
				 * write xlog.
				 */
				RepGetBytes((char *) recvBuf, len, repConn);
				RepPutMessage(repConn, 'r', (const void *) &recptr, 8);
				RepXLogWrite(WriteRqst, len);

				break;
			}

			case 'E':	/* error */
				fprintf(stderr, _("%s: error occured in server: %s\n"),
						progname, PQerrorMessage(repConn));
				RepExit(1);
				
			default :	/* ignore */
			{
				if (PQdiscardBytes(len, repConn) == EOF)
				{
					fprintf(stderr, _("%s: unexpected EOF on connection: %s\n"),
							progname, PQerrorMessage(repConn));
					RepExit(1);
				}
				
				fprintf(stderr, _("%s: message type (%c) ignored: %s\n"),
						progname, qtype, PQerrorMessage(repConn));
			}
		}
	}

	/* can't get here because the above loop never exits */
	return 1;
}

/*
 * Exit from walreceiver
 */
static void
RepExit(int status)
{
	PQfinish(repConn);
	exit(status);
}


/*
 * signal handler routines
 */

/* SIGINT, SIGTERM, SIGQUIT: exit normally */
static void
RepExitHandler(SIGNAL_ARGS)
{
	RepExit(0);
}


/*
 * routines for I/O
 */

/*
 * Create a new xlog file segment. 
 *
 * This function is based heaviliy on XLogFileInit and InstallXLogFileSegment.
 *
 * log, seg: identify segment to be created/opened.
 *
 * Returns FD of opened file.
 */
static int
RepXLogFileInit(uint32 log, uint32 seg)
{
	char	path[MAXPGPATH];
	char	tmppath[MAXPGPATH];
	char   *zbuffer;
	int		fd;
	int		nbytes;
	
	RepXLogFilePath(path, xlogLoc, repTLI, log, seg);
	
	/* Initialize an empty (all zeroes) segment */
	snprintf(tmppath, MAXPGPATH, "%s/xlogtemp.%d", xlogLoc, (int) getpid());
	
	unlink(tmppath);

	fd = open(tmppath, O_RDWR | O_CREAT | O_EXCL | PG_BINARY, S_IRUSR | S_IWUSR);
	if (fd < 0)
	{
		fprintf(stderr, _("%s: could not create log file %u, segment %u: %s\n"),
				progname, writeId, writeSeg, strerror(errno));
		RepExit(1);
	}
	
	/*
	 * Zero-fill the file.
	 */
	zbuffer = (char *) calloc(XLOG_BLCKSZ, sizeof(char));
	for (nbytes = 0; nbytes < XLogSegSize; nbytes += XLOG_BLCKSZ)
	{
		errno = 0;
		if ((int) write(fd, zbuffer, XLOG_BLCKSZ) != (int) XLOG_BLCKSZ)
		{
			/* if write didn't set errno, assume no disk space */
			if (errno == 0)
				errno = ENOSPC;
			fprintf(stderr, _("%s: could not write to log file %u, segment %u: "
							  "%s\n"),
					progname, writeId, writeSeg, strerror(errno));
			RepExit(1);
		}
	}
	free(zbuffer);
	
	if (fsync(fd) != 0)
	{
		fprintf(stderr, _("%s: could not fsync log file %u, segment %u: %s\n"),
				progname, writeId, writeSeg, strerror(errno));
		RepExit(1);
	}

	if (close(fd))
	{
		fprintf(stderr, _("%s: could not close log file %u, segment %u: %s\n"),
				progname, writeId, writeSeg, strerror(errno));
		RepExit(1);
	}

	/* Get rid of any pre-existing segment file */
	unlink(path);

	if (rename(tmppath, path) < 0)
	{
		fprintf(stderr, _("%s: could not rename file \"%s\" to \"%s\" "
						  "(initialization of log file %u, segment %u): %s\n"),
				progname, tmppath, path, log, seg, strerror(errno));
		RepExit(1);
	}

	/* Now open original target segment */
	fd = open(path, O_RDWR | PG_BINARY, S_IRUSR | S_IWUSR);
	if (fd < 0)
	{
		fprintf(stderr, _("%s: could not open log file %u, segment %u: %s\n"),
				progname, writeId, writeSeg, strerror(errno));
		exit(1);
	}
	
	return fd;
}

/*
 * Open a pre-existing logfile segment for writing.
 *
 * This is heavily based on XLogFileOpen.
 */
static int
RepXLogFileOpen(uint32 log, uint32 seg)
{
	char	path[MAXPGPATH];
	int		fd;

	RepXLogFilePath(path, xlogLoc, repTLI, log, seg);

	fd = open(path, O_RDWR | PG_BINARY, S_IRUSR | S_IWUSR);
	if (fd < 0)
	{
		fprintf(stderr, _("%s: could not open log file %u, segment %u: %s\n"),
				progname, writeId, writeSeg, strerror(errno));
		RepExit(1);
	}
	
	return fd;
}

/*
 * Close the current logfile segment for writing.
 *
 * This is based on XLogFileClose.
 */
static void
RepXLogFileClose(void)
{
	if (close(writeFile))
	{
		fprintf(stderr, _("%s: could not close log file %u, segment %u: %s\n"),
				progname, writeId, writeSeg, strerror(errno));
		RepExit(1);
	}
	writeFile = -1;

	if (RepXLogArchivingActive())
		RepXLogArchive();
}

/*
 * Write and/or fsync the log at least as far as WriteRqst indicates.
 *
 * This is based on XLogWrite.
 */
static void
RepXLogWrite(XLogRecPtr WriteRqst, int32 nbytes)
{
	uint32	startoffset;

	if (!XLByteInPrevSeg(WriteRqst, writeId, writeSeg))
	{
		/*
		 * Switch to new logfile segment.  We cannot have any pending
		 * pages here (since we dump what we have at segment end).
		 */
		if (writeFile >= 0)
			RepXLogFileClose();
		XLByteToPrevSeg(WriteRqst, writeId, writeSeg);
		
		/* create/use new log file */
		writeFile = RepXLogFileInit(writeId, writeSeg);
		writeOff = 0;
	}
	
	/* Make sure we have the current logfile open */
	if (writeFile < 0)
	{
		XLByteToPrevSeg(WriteRqst, writeId, writeSeg);
		writeFile = RepXLogFileOpen(writeId, writeSeg);
		writeOff = 0;
	}
	
	startoffset = WriteRqst.xrecoff % XLogSegSize;
	startoffset = ((startoffset == 0) ? XLogSegSize : startoffset) - nbytes;
	
	/* Need to seek in the file? */
	if (writeOff != startoffset)
	{
		if (lseek(writeFile, (off_t) startoffset, SEEK_SET) < 0)
		{
			fprintf(stderr, _("%s: could not seek in log file %u, "
							  "segment %u to offset %u: %s\n"),
					progname, writeId, writeSeg, startoffset, strerror(errno));
			exit(1);
		}
		writeOff = startoffset;
	}
	
	/* OK to write */
	if (write(writeFile, recvBuf, nbytes) != nbytes)
	{
		/* if write didn't set errno, assume no disk space */
		if (errno == 0)
			errno = ENOSPC;
		fprintf(stderr, _("%s: could not write to log file %u, segment %u "
						  "at offset %u, length %lu: %s"),
				progname, writeId, writeSeg, writeOff,
				(unsigned long) nbytes, strerror(errno));
		RepExit(1);
	}
}

/*
 * Move filled xlog file segment to the archive location.
 */
static void
RepXLogArchive(void)
{
	char xlogPath[MAXPGPATH];
	char archPath[MAXPGPATH];

	RepXLogFilePath(xlogPath, xlogLoc, repTLI, writeId, writeSeg);
	RepXLogFilePath(archPath, archLoc, repTLI, writeId, writeSeg);
	
	if (rename(xlogPath, archPath) < 0)
	{
		fprintf(stderr, _("%s: could not rename file \"%s\" to \"%s\" "
						  "(initialization of log file %u, segment %u): %s\n"),
				progname, xlogPath, archPath, writeId, writeSeg, strerror(errno));
		RepExit(1);
	}
}


/*
 * routines for communication
 */

/*
 * Set up a connection for replication.
 *
 * This is heavily based on doConnect in contrib/pgbench/pgbench.c.
 */
static PGconn *
RepConnect(void)
{
	PGconn *conn;
	bool	new_pass;
	
	/* Loop until we have a password if requested */
	do
	{
		new_pass = false;
		conn = PQsetdbLogin(repHost, repPort, repOptions, repTty,
							repDbName, repLogin, repPwd);
		
		if (PQstatus(conn) == CONNECTION_BAD)
		{
			fprintf(stderr, _("%s: failed to connect to \"%s\": %s\n"),
					progname, repDbName, PQerrorMessage(conn));
			RepExit(1);
		}
		
		if (PQstatus(conn) == CONNECTION_BAD &&
			PQconnectionNeedsPassword(conn) &&
			repPwd == NULL &&
			!feof(stdin))
		{
			PQfinish(conn);
			repPwd = simple_prompt("Password", 100, false);
			new_pass = true;
		}
	} while (new_pass);
	
	if (PQstatus(conn) == CONNECTION_BAD)
	{
		fprintf(stderr, _("%s: failed to connect to \"%s\": %s\n"),
				progname, repDbName, PQerrorMessage(conn));
		RepExit(1);
	}

	return conn;
}

/*
 * Get a known number of bytes from connection and handle error.
 */
static void
RepGetBytes(char *s, size_t len, PGconn *conn)
{
	if (PQgetBytes(s, len, conn) == EOF)
	{
		fprintf(stderr, _("%s: unexpected EOF on connection: %s\n"),
				progname, PQerrorMessage(repConn));
		RepExit(1);
	}
}

/*
 * Send a message to server.
 */
static void
RepPutMessage(PGconn *conn, char pack_type,
			  const void *buf, size_t buf_len)
{
	if (PQputMessage(conn, pack_type, buf, buf_len) == STATUS_ERROR)
	{
		fprintf(stderr, _("%s: could not send a message (%d): %s"),
				progname, pack_type, PQerrorMessage(repConn));
		RepExit(1);
	}	
}


/*
 * common routines
 */

/*
 * Make sure input buffer can hold bytes_needed bytes (caller must
 * include already-stored data into the value!)
 *
 * This function is based heavily on pqCheckInBufferSpace.
 *
 * Returns 0 on success, EOF if failed to enlarge buffer
 */
static int
RepCheckRecvBufSpace(size_t bytes_needed)
{
	int		newsize = bufSize;
	char   *newbuf;

	if (bytes_needed <= (size_t) newsize)
		return 0;
	
	/*
	 * If we need to enlarge the buffer, we first try to double it in size; if
	 * that doesn't work, enlarge in multiples of 8K.  This avoids thrashing
	 * the malloc pool by repeated small enlargements.
	 *
	 * Note: tests for newsize > 0 are to catch integer overflow.
	 */
	do
	{
		newsize *= 2;
	} while (newsize > 0 && bytes_needed > (size_t) newsize);
	
	if (newsize > 0 && bytes_needed <= (size_t) newsize)
	{
		newbuf = realloc(recvBuf, newsize);
		if (newbuf)
		{
			/* realloc succeeded */
			recvBuf = newbuf;
			bufSize = newsize;
			return 0;
		}
	}

	newsize = bufSize;
	do
	{
		newsize += 8192;
	} while (newsize > 0 && bytes_needed > (size_t) newsize);

	if (newsize > 0 && bytes_needed <= (size_t) newsize)
	{
		newbuf = realloc(recvBuf, newsize);
		if (newbuf)
		{
			/* realloc succeeded */
			recvBuf = newbuf;
			bufSize = newsize;
			return 0;
		}
	}

	/* realloc failed. Probably out of memory */
	return EOF;
}

/*
 * Show usage
 */
static void
usage(void)
{
	fprintf(stderr, _("Usage:\n"));
	fprintf(stderr, _("  walreceiver [OPTION]... [XLOGLOCATION] [ARCHIVELOCATION]\n"));
	fprintf(stderr, _("\nOptions:\n"));
	fprintf(stderr, _("  -h HOSTNAME	database server host or socket directory "
					  "(default: local socket)\n"));
	fprintf(stderr, _("  -p PORT	database server port (default: \"%s\")\n"),
			DEF_PGPORT_STR);
	fprintf(stderr, _("  -U NAME	database user name (default: postgres)\n"));
	fprintf(stderr, _("  -?		show usage\n"));
	fflush(stderr);
}
