import java.util.*;
import java.net.*;
import java.sql.*;
import java.io.*;
import org.postgresql.util.*;

/*
	1) Edit the code and set it to connect to your test db
	2) Run this as "java SetGetObject init" to make the myurl table
	3) then, create another table:
		CREATE TABLE website (urlOID oid, url myurl);
	4) Then, run it again without "init".

	Note that the field type is "myurl" and is the name of
	the table that the java class is stored in. This makes
	use of a postgres user-defined type capability that
	lets a table act as a data type.  This functionality
	in postgres in incomplete or might even be broken
	if you try the wrong query.

	The oid of the myurl row is also
	stored in website.  This duplicates data since both
	oid and myurl are same number, pointing to a row
	in the myurl table.  However, this is necessary because
	postgres' support for using a table as a datatype is
	imcomplete/broken and you cannot compare or access
	the myurl datatype as you would like. For example,
	no = operator will work on them and you cannot
	access elements using multiple levels of . notation
	either.  To be able to still access them easily,
	you need to keep a handle on the row oid an as OID
	datatype too.  You might ask why not just use only the oid and not
	bother with the myurl field type? Well, the jdbc
	driver will treat an OID type as just a long integer, but
	it will see the other types like myurl as serialize types
	and call Serialize.fetch for them.  If only the OID type
	is available as a reference to the stored java object,
	then the driver would not know when you just want the oid
	itself or when you want the stored java object.  By
	having both in the table, the types are different and
	getObject will know which you are wanting to get.
	Again, due to the limited support for the table-type
	fields, the oid is your real handle on the object
	for doing selects to locate them.

*/

public class SetGetObject
{
	public static void main( String[] args ) {
		try {
			Class.forName("org.postgresql.Driver");
		} catch( ClassNotFoundException e ) {
			System.out.println("ClassNotFoundException loading pg driver: " + e.getMessage() );
			return;
		}

		try {
			myurl url = new myurl("http://www.comptechnews.com/");
			System.out.println("URL to be stored: ");
			printURL( url );

			DriverManager.setLogWriter( new PrintWriter(System.out) );
			Connection db = DriverManager.getConnection("jdbc:postgresql:setgetobject", "pgdevel", "pgdevel");
			System.out.println("Loaded PostgreSQL JDBC Driver.");

			Serialize.create( (org.postgresql.Connection) db, myurl.class );
			System.out.println("Serialize.create() called Ok.");
			if( args.length > 0 ) return;

			PreparedStatement ps = db.prepareStatement("INSERT INTO website (urlOID, url) VALUES ( ?, ? )");
			ps.setObject( 2, url );
			ps.setInt( 1, url.oid );
			// NOTE that the ORDER of the above setxxx statements matter!
			// In general, the setObject calls should be done 1st, because
			// setObject on a java class will update the object (url) and set its
			// oid field, which is then used in the next setString call.
			// Initially, url.oid is 0, but after setObject, it is serialized into the databse
			// and its oid field is updated.

			System.out.println( ps.toString() );
			System.out.println("URLs INSERTed: " + ps.executeUpdate() );

			BufferedReader in = new BufferedReader( new InputStreamReader(System.in) );
			in.readLine();

			// Because postgres support for using a table as a datatype is broken/incomplete,
			// queries on the oid::tablename type has little or no support and often will result
			// in fmgr_info xxxx: cache lookup failed messages.  You have to access the objects
			// thru the table that holds the oid::tablename references.
			PreparedStatement ps2 = db.prepareStatement("SELECT url FROM website WHERE urlOID = ? " );
			ps2.setInt( 1, url.oid );
			System.out.println( ps2.toString() );
			ResultSet rs = ps2.executeQuery();
			myurl gotURL;
			if( rs.next() ) {
				gotURL = (myurl) rs.getObject("url");
				System.out.println("URL retrieved: ");
				printURL( gotURL );
			}
			else System.out.println("URL was not retrieved.");

			rs.close();
			ps2.close();
			ps.close();
			db.close();

   		} catch( SQLException sqle ) {
			System.out.println("SQLException: " + sqle.getMessage() );
			sqle.printStackTrace();
			return;

		} catch( IOException ioe ) {
			System.out.println("IOException: " + ioe.getMessage() );
			ioe.printStackTrace();
			return;
		} catch(Exception e) {
			System.out.println("Exception: " + e.getMessage() );
			e.printStackTrace();
			return;
		}


	}

	public static void printURL( myurl u ) {
		try {
			URL t = new URL( u.url );
			System.out.println("URL: " + t.toString() );
			System.out.println("\tProto: " + t.getProtocol() );
			System.out.println("\tHost: " + t.getHost() );
			System.out.println("\tPath: " + t.getPath() );
			System.out.println("\toid: " + u.oid );
		} catch( MalformedURLException mfe ) {}

	}
}

